import json
import base64
import time
import requests
import os
import uuid
import hashlib
import random
import string
import mechanicalsoup
from urllib.parse import urlparse, parse_qs
import datetime
import re

class APIParser():
    class Season():
        def __init__(self, id, title, numberOfEpisodes, seasonType, year=None, month=None, no=None):
            self.id = id
            self.title = title
            self.numberOfEpisodes = numberOfEpisodes
            self.seasonType = seasonType
            self.year = year
            self.month = month
            self.no = no
        def __str__(self):
            return f"ID: {self.id}, Title: {self.title}, numberOfEpisodes {self.numberOfEpisodes}"

    class RecommendationOverview():
        def __init__(self, id, title):
            self.id = id
            self.title = title

        def __str__(self):
            return f"ID: {self.id}, Title: {self.title}"

    class Event():
        def __init__(self, id, title, tier, description, startTimestamp, landscapeImg, portraitImage):
            self.id = id
            self.title = title
            self.tier = tier
            self.startTimestamp = startTimestamp
            self.description = description
            self.landscapeImg = landscapeImg
            self.portraitImage = portraitImage
        def __str__(self):
            return f"ID: {self.id}, Title: {self.title}, Description: {self.description}, Start Timestamp: {self.startTimestamp}, Tier {self.tier}, Landscape Image: {self.landscapeImg}, Portrait Image: {self.portraitImage}"

    class FormatOverview():
        def __init__(self, id, title, tier, formatType, landscapeImg, portraitImage, description=None):
            self.id = id
            self.title = title
            self.tier = tier
            self.formatType = formatType
            self.description = description
            self.landscapeImg = landscapeImg
            self.portraitImage = portraitImage
        def __str__(self):
            return f"ID: {self.id}, Title: {self.title}, Description: {self.description}, Tier {self.tier}, Landscape Image: {self.landscapeImg}, Portrait Image: {self.portraitImage}"

    class FormatDescription():
        def __init__(self, id, title, tier, description, productionYear, productionCountries, numberOfSeasons, numberOfEpisodes, genres, seasons, episodeNo, season, duration, ageRating, landscapeImg, portraitImage):
            self.id = id
            self.title = title
            self.tier = tier
            self.description =description
            self.productionYear = productionYear
            self.productionCountries = productionCountries
            self.numberOfSeasons = numberOfSeasons
            self.numberOfEpisodes = numberOfEpisodes
            self.genres = genres
            self.seasons = seasons
            self.episodeNo = episodeNo
            self.season = season
            self.duration = duration
            self.ageRating = ageRating
            self.landscapeImg = landscapeImg
            self.portraitImage = portraitImage
        def __str__(self):
            return f"ID: {self.id}, Title: {self.title}, Tier {self.tier}, Description: {self.description}, Production Year: {self.productionYear}, " \
                   f"Production Countries: {self.productionCountries}, Number of Episodes: {self.numberOfEpisodes}, Number of Seasons: {self.numberOfSeasons}, " \
                   f"Genres: {self.genres}, Seasons: {self.seasons}, Landscape Image: {self.landscapeImg}, Portrait Image: {self.portraitImage}"

    @staticmethod
    def getElement(json, path : list, allowNone : bool = False ):
        while json != None and len(path) > 0:
            json = json.get(path[0])
            del path[0]
        if not allowNone and json == None:
            raise Exception("Element not found")
        return json

    def parseFormatDetailed(json) -> FormatDescription:
        id = APIParser.getElement(json, ["id"])
        title = APIParser.getElement(json, ["title"])
        tier = APIParser.getElement(json, ["tier"], True)
        description = APIParser.getElement(json, ["emptyFormatText"], True)
        if description == None:
            description = APIParser.getElement(json, ["descriptionV2"], True)
        productionYear = APIParser.getElement(json, ["productionYear"], True)
        productionCountries = APIParser.getElement(json, ["productionCountries"], True)
        numberOfSeasons = APIParser.getElement(json, ["numberOfSeasons"], True)
        numberOfEpisodes = APIParser.getElement(json, ["numberOfEpisodes"], True)
        episodeNo = APIParser.getElement(json, ["number"], True)
        duration = APIParser.getElement(json, ["durationInSecondsV2"], True)
        ageRating = APIParser.getElement(json, ["ageRating"], True)
        genres = APIParser.getElement(json, ["genres"], True)
        landscapeImg = APIParser.getElement(json, ["watchImages", "artworkLandscape", "absoluteUri"], True)
        portraitImage = APIParser.getElement(json, ["watchImages", "plainLandscape", "absoluteUri"], True)
        if portraitImage == None:
            portraitImage = APIParser.getElement(json, ["watchImages", "artworkPortrait", "absoluteUri"], True)
        if landscapeImg == None:
            landscapeImg = APIParser.getElement(json, ["watchImages", "default", "absoluteUri"], True)
        seasons = APIParser.parseSeasons(APIParser.getElement(json, ["seasons"], True))
        season = APIParser.getElement(json, ["episodeSeason", "season", "ordinal"], True)
        return APIParser.FormatDescription(id, title, tier, description, productionYear, productionCountries, numberOfSeasons, numberOfEpisodes, genres, seasons, episodeNo, season, duration, ageRating, landscapeImg, portraitImage)

    def parseFormatOverview(json) -> FormatOverview:
        id = APIParser.getElement(json, ["id"])
        title = APIParser.getElement(json, ["title"])
        tier = APIParser.getElement(json, ["tier"], True)
        formatType = APIParser.getElement(json, ["__typename"])
        landscapeImg = APIParser.getElement(json, ["watchImages", "artworkLandscape", "absoluteUri"], True)
        if landscapeImg == None:
            landscapeImg = APIParser.getElement(json, ["images", "artworkLandscape", "url"], True)
        portraitImage = APIParser.getElement(json, ["watchImages", "artworkPortrait", "absoluteUri"], True)
        if portraitImage == None:
            portraitImage = APIParser.getElement(json, ["images", "artworkPortrait", "url"], True)
        description = APIParser.getElement(json, ["urlData", "seo", "description"], True)
        return APIParser.FormatOverview(id, title, tier, formatType, landscapeImg, portraitImage, description)

    def parseEvent(json) -> Event:
        id = APIParser.getElement(json, ["id"])
        title = APIParser.getElement(json, ["title"])
        tier = APIParser.getElement(json, ["tier"], True)
        description = APIParser.getElement(json, ["urlData", "seo", "description"], True)
        streamStart = APIParser.getElement(json, ["streamStart"], True)
        startTimestamp = 0
        if streamStart != None:
            cleanTimeStr = str(streamStart).split('.')[0]
            format = '%Y-%m-%dT%H:%M:%S'
            try:
                dateTimeObj = datetime.datetime.strptime(cleanTimeStr, format)
            except TypeError:
                dateTimeObj = datetime.datetime(*(time.strptime(cleanTimeStr, format)[0:6]))
            startTimestamp = time.mktime(dateTimeObj.timetuple())
        landscapeImg = APIParser.getElement(json, ["watchImages", "artworkLandscape", "absoluteUri"], True)
        if landscapeImg == None:
            landscapeImg = APIParser.getElement(json, ["images", "artworkLandscape", "url"], True)
        portraitImage = APIParser.getElement(json, ["watchImages", "artworkPortrait", "absoluteUri"], True)
        if portraitImage == None:
            portraitImage = APIParser.getElement(json, ["images", "artworkPortrait", "url"], True)

        return APIParser.Event(id, title, tier, description, startTimestamp, landscapeImg, portraitImage)

    def parseEpisodes(json) -> list:
        elements = APIParser.getElement(json, ["data", "season", "episodes"])
        episodes = []
        for el in elements:
            episodes.append(APIParser.parseFormatDetailed(el))
        return episodes

    def parseSeasons(json) -> list:
        seasons = []
        if json == None:
            return []
        for season in json:
            id = APIParser.getElement(season, ["id"])
            numberOfEpisodes = APIParser.getElement(season, ["numberOfEpisodes"], True)
            seasonType = APIParser.getElement(season, ["seasonType"], True)
            month = APIParser.getElement(season, ["month"], True)
            year = APIParser.getElement(season, ["year"], True)
            ordinal = APIParser.getElement(season, ["ordinal"], True)

            if seasonType == "ANNUAL":
                title = f'{month:02d}.{year:04d}'
            else:
                title = f'Staffel {ordinal}'

            seasons.append(APIParser.Season(id, title, numberOfEpisodes, seasonType, year, month, ordinal))
        return seasons

    def parseFormats(json) -> list:
        formats = []
        for el in json:
            formats.append(APIParser.parseFormatOverview(el))
        return formats

    def parseLiveChannel(json) -> list:
        id = APIParser.getElement(json, ["id"])
        title = APIParser.getElement(json, ["name"])
        tier = "PREMIUM"
        return APIParser.FormatOverview(id, title, tier, "LIVE", None, None)

    def parseLiveChannels(json) -> list:
        elements = APIParser.getElement(json, ["data", "liveTvStations"])
        formats = []
        for el in elements:
            formats.append(APIParser.parseLiveChannel(el))
        return formats

    def parseEvents(json) -> list:
        teaserRows = APIParser.getElement(json, ["data", "liveEventsOverview", "teaserRows"])
        events = []
        for row in teaserRows:
            for event in APIParser.getElement(row, ["events"]):
                events.append(APIParser.parseEvent(event))
        return events

class API():
    USERAGENT = "Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:109.0) Gecko/20100101 Firefox/118.0"
    ENDPOINTS = {
        'Format' : '63361bdda2780638dcc2e971ae4233df4b0db634aca1bdd0a37b713108326778',
        'SeasonWithFormatAndEpisodes' : '8f6f7eeb8dbc1c8e22e7488c209f4c19dbc2ef179efec936214f807dc1a11503',
        'MRE' : '8f6f7eeb8dbc1c8e22e7488c209f4c19dbc2ef179efec936214f807dc1a11503',
        'Recommendations' : '11ec41adc09e1ae2775b6f965ffeba1401928b9c3c7223ce57fd099525b57a17',
        'OverviewPage' : '28aad4e992bb63330bfcd40a6906af3119d8a2612fa9fd28dae9c19127e247ca',
        'LiveTvStations' : 'd74c69c5a7e403f947a3c3ff09022665ad97f4c3ff3f837b1b202c24ea0235f5',
        'LiveEventsOverviewPage' : '8bd2e224e31a7914d299a032b779c32c7c8a6e430949247dadd75cb673cf61e9',
        'MovieDetail' : '8ec9544a5c3041fc6e3ed3ae96f858505abc454bd18a7b2b6110e30643f9cd49',
        'WatchPlayerConfigV2' : '4fb3276d6b919af22dd24b6411ae8f62e3fa89d37e5de33fa3f6f1327a97e554',
        'Search' : 'dda4ff255c64fc3a7dd992813129fee40e1c95334d5a66d002ea325eeb566dd3',
        'ExploreWidgetWatch' : '19110e219a2606ecbb13f2daec2450c1f1895ee70ac986a508d89cd320d738cb'

    }
    APIBASEURL = base64.b64decode(b'aHR0cHM6Ly9jZG4uZ2F0ZXdheS5ub3ctcGx1cy1wcm9kLmF3cy1jYmMuY2xvdWQvZ3JhcGhxbA==').decode()
    AUTHBASEURL = base64.b64decode(b'aHR0cHM6Ly9hdXRoLnJ0bC5kZS9hdXRoL3JlYWxtcy9ydGxwbHVzL3Byb3RvY29sL29wZW5pZC1jb25uZWN0').decode()

    def postRequest(self, url : str, postData : dict=None, jsonData : json = None, request : requests.Session = None ) -> requests.Response:
        headers = {
            'User-Agent': API.USERAGENT,
            'Referer' :  base64.b64decode(b'aHR0cHM6Ly9wbHVzLnJ0bC5kZS8=').decode(),
            'Origin' :  base64.b64decode(b'aHR0cHM6Ly9wbHVzLnJ0bC5kZS8=').decode(),
            'Content-Type' : "application/x-www-form-urlencoded"
        }
        if (postData == None and jsonData == None) or (postData != None and jsonData != None):
            return None
        if request != None:
            return request.post(url, data=postData, json=jsonData, headers=headers)
        else:
            return requests.post(url, data=postData, json=jsonData, headers=headers)

    def getRequest(self, url: str, token=None, client=True) -> requests.Response:
        headers = {
            'User-Agent': API.USERAGENT,
            'Referer' :  base64.b64decode(b'aHR0cHM6Ly9wbHVzLnJ0bC5kZS8=').decode(),
            'Origin' :  base64.b64decode(b'aHR0cHM6Ly9wbHVzLnJ0bC5kZS8=').decode(),
        }
        if token != None:
            headers["Authorization"] = f"Bearer {token}"
        if client:
            headers[base64.b64decode(b'cnRscGx1cy1jbGllbnQtSWQ=').decode()] = base64.b64decode(b'cmNpOnJ0bHBsdXM6d2Vi').decode()
            headers[base64.b64decode(b'cnRscGx1cy1jbGllbnQtVmVyc2lvbg==').decode()] = self.clientVersion
        res = requests.get(url, headers=headers)
        if res.status_code != 200:
            print(res.text)
            raise Exception("Invalid Response")
        return res

    @staticmethod
    def isTokenValid(token : str) -> bool:
        valid = True
        try:
            base64Parts = token.split(".")
            token = "%s==" % base64Parts[1]
            userData = json.loads(base64.b64decode(token))

            if "licenceEndDate" in userData:
                licenceEndDate = userData["licenceEndDate"].split("+")[4]
                try:
                    format = '%Y-%m-%dT%H:%M:%S'
                    try:
                        licenceEndDateTS = datetime.datetime.strptime(licenceEndDate, format)
                    except TypeError:
                        licenceEndDateTS = datetime.datetime(*(time.strptime(licenceEndDate, format)[0:6]))
                    licenceEndDateTS = time.mktime(licenceEndDateTS)
                    if licenceEndDateTS < (time.time() + 60):
                        valid = False
                except:
                    valid = False
            if (valid and "exp" in userData and
                userData["exp"] > (time.time() + 60)):
                valid = True
            else:
                valid = False
        except:
            valid = False
        return valid

    @staticmethod
    def parseTokenConfig(config):
        postData = {}
        for pair in config.split(","):
            splittedPair = pair.split(":")
            postData[splittedPair[0]] = splittedPair[1].replace('"','')
        return postData

    @staticmethod
    def getConfig():
        baseEndPoint = base64.b64decode(b'aHR0cHM6Ly9wbHVzLnJ0bC5kZS8=').decode()
        endPoint = baseEndPoint
        headers = {"User-Agent": API.USERAGENT}
        r = requests.get(endPoint,headers=headers)
        try:
            jsName = re.findall(r'<script src="(main[A-z0-9\-\.]+\.js)"', r.text, re.S)[-1]
        except:
            return None
        endPoint = baseEndPoint + jsName
        r = requests.get(endPoint,headers=headers)
        return r.text

    @staticmethod
    def getClientVersion():
        baseEndPoint = base64.b64decode(b'aHR0cHM6Ly9wbHVzLnJ0bC5kZS9hc3NldHMvY29uZmlnL2NvbmZpZy5qc29u=').decode()
        endPoint = baseEndPoint
        headers = {"User-Agent": API.USERAGENT}
        r = requests.get(endPoint,headers=headers)
        try:
            return r.json()["version"]
        except:
            return None

    @staticmethod
    def getClientID() -> str:
        text = API.getConfig()
        m = re.search(r'clientId:"([^"]+)"', text)
        if m:
            return m.group(1)
        return None

    def receiveToken(self) -> str:
        text = API.getConfig()
        m = re.search(r'anonymousCredentials:{([^}]+)}', text)
        if m:
            authURL = API.AUTHBASEURL + "/token"
            postData = API.parseTokenConfig(m.group(1))
            res = self.postRequest(authURL, postData=postData)
            if res.status_code == 200:
                token = res.json().get("access_token")
                return token
        return None

    def renewToken(self):
        authURL = API.AUTHBASEURL + "token?ngsw-bypass="
        postData = {
                "grant_type": "refresh_token",
                "refresh_token": self.refreshToken,
                "client_id": self.clientID
        }
        res = self.postRequest(authURL, postData=postData)
        if res.status_code == 200:
            token = res.json().get("access_token")
            refresh_token = res.json().get("refresh_token")
            return token, refresh_token
        else:
            print(res.text)
        return None, None

    def getToken(self):
        return self.token

    def getRefreshToken(self):
        return self.refreshToken

    def __init__(self, token=None, refresh_token=None, clientID=None, clientVersion=None):
        self.refreshToken = refresh_token
        self.token = None
        if clientID == None:
            self.clientID = API.getClientID()
        else:
            self.clientID = clientID
        if clientVersion == None:
            self.clientVersion = API.getClientVersion()
        else:
            self.clientVersion = clientVersion
        if token == None or token == "":
            token = self.receiveToken()
            if self.isTokenValid(token):
                self.token = token
        else:
            if self.isTokenValid(token):
                self.token = token
            elif self.refreshToken != None and self.refreshToken != "":
                self.token, self.refreshToken = self.renewToken()
            else:
                self.token = self.receiveToken()
        if self.token == None:
            raise Exception("no token")

    class Login():
        def __init__(self, api):
            self.api = api
            code_verifier = ''.join(random.SystemRandom().choice(string.ascii_letters + string.digits) for _ in range(96))
            hashed = hashlib.sha256(code_verifier.encode('ascii')).digest()
            encoded = base64.urlsafe_b64encode(hashed)
            code_challenge = encoded.decode('ascii')[:-1]
            state = uuid.uuid4()
            nonce = uuid.uuid4()
            reqURL = f'{API.AUTHBASEURL}/auth?client_id={self.api.clientID}&redirect_uri=https%3A%2F%2F{base64.b64decode(b"cGx1cy5ydGwuZGU==").decode()}%2F&state={state}&response_mode=query&response_type=code&scope=openid%20email&nonce={nonce}&prompt=login&code_challenge={code_challenge}&code_challenge_method=S256'
            browser = mechanicalsoup.StatefulBrowser(soup_config={'features': 'html.parser'})
            browser.set_user_agent("Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:109.0) Gecko/20100101 Firefox/118.0")
            browser.open(reqURL)
            self.browser = browser
            self.code_verifier = code_verifier
            self.access_token = None
            self.refresh_token = None


        def sendLogin(self, username, password):
            if self.browser == None or self.api.clientID == None or self.code_verifier == None:
                return False
            self.browser.select_form('form')
            self.browser["username"] = username,
            self.browser["password"] = password
            response = self.browser.submit_selected()

            request = requests.Session()
            request.cookies = response.history[0].cookies
            parsed_url = urlparse(response.url)
            captured_value = parse_qs(parsed_url.query)['code'][0]
            data = {
                "code": captured_value,
                "grant_type": "authorization_code",
                "client_id": self.api.clientID,
                "redirect_uri": base64.b64decode(b'aHR0cHM6Ly9wbHVzLnJ0bC5kZS8=').decode(),
                "code_verifier": self.code_verifier
            }

            authURL = f"{API.AUTHBASEURL}/token?ngsw-bypass="
            res = self.api.postRequest(authURL, postData=data, request=request)
            if res.status_code == 200:
                try:
                    data = res.json()
                    self.access_token = data["access_token"]
                    self.refresh_token = data["refresh_token"]
                    self.id_token = data["id_token"]
                except:
                    print(res.text)
                    return False
                return True
            else:
                print(res.text)
            return False

        def getAccessToken(self):
            return self.access_token

        def getRefreshToken(self):
            return self.refresh_token

    def getFormat(self, id):
        requestURL = f'{API.APIBASEURL}?operationName=Format&variables={{"id": "{id}" }}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{API.ENDPOINTS["Format"]}"}}}}'
        res = self.getRequest(requestURL, self.token)
        jsonData = res.json()
        format = APIParser.parseFormatDetailed(APIParser.getElement(jsonData, ["data", "format"]))
        return format

    def getSeason(self, seasonId):
        offset = 0
        limit = 5
        episodes = []
        while True:
            requestURL = f'{API.APIBASEURL}?operationName=SeasonWithFormatAndEpisodes&variables={{"seasonId": "{seasonId}","offset":{offset},"limit":{limit} }}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{API.ENDPOINTS["SeasonWithFormatAndEpisodes"]}"}}}}'
            res = self.getRequest(requestURL, self.token)
            jsonData = res.json()
            episodes += APIParser.parseEpisodes(jsonData)
            totalItems = APIParser.getElement(jsonData, ["data", "season", "numberOfEpisodes"])
            offset += limit
            if offset >= totalItems:
                break
        return episodes

    def getMovie(self, id):
        requestURL = f'{API.APIBASEURL}?operationName=MovieDetail&variables={{"id": "{id}" }}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{API.ENDPOINTS["MovieDetail"]}"}}}}'
        res = self.getRequest(requestURL, self.token)
        jsonData = res.json()
        format = APIParser.parseFormatDetailed(APIParser.getElement(jsonData, ["data", "movie"]))
        return format

    def _parseManifest(self, jsonData):
        playbackURL = APIParser.getElement(jsonData, ["videoUrl"])
        licenseURL = None
        for license in APIParser.getElement(jsonData, ["licenses"]):
            licenseType = APIParser.getElement(license, ["type"])
            if licenseType == "WIDEVINE":
                licenseURL = APIParser.getElement(license, ["licenseUrl"])
        return playbackURL, licenseURL

    def getPlaybackURL(self, id):
        requestURL = f'{API.APIBASEURL}?operationName=WatchPlayerConfigV2&variables={{"platform":"WEB","id": "{id}" }}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{API.ENDPOINTS["WatchPlayerConfigV2"]}"}}}}'
        res = self.getRequest(requestURL, self.token)
        jsonData = res.json()
        playoutVariants = APIParser.getElement(jsonData, ["data", "watchPlayerConfigV2", "playoutVariants"])
        playbackURL = None
        licenseURL = None
        for variant in playoutVariants:
            manifestType = APIParser.getElement(variant, ["type"])
            if manifestType == "dashhd":
                playbackURL, licenseURL = self._parseManifest(variant)
                break
            if manifestType == "dashsd":
                playbackURL, licenseURL = self._parseManifest(variant)
        return playbackURL, licenseURL

    def getOverview(self, elementType):
        limit = 48
        offset = 0
        apiParams = {
            'APIBASEURL' : API.APIBASEURL,
            'offset' : offset,
            'limit' : limit,
            'elementType' : elementType,
            'endpoint' : API.ENDPOINTS["OverviewPage"]
        }
        requestURL = '{APIBASEURL}?operationName=OverviewPage&variables={{"pagination":{{"offset":{offset}, "limit":{limit}}},"filter":{{"elementType":"{elementType}"}}}}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{endpoint}"}}}}'
        res = self.getRequest(requestURL.format(**apiParams), self.token)
        jsonData = res.json()
        formats = APIParser.parseFormats(APIParser.getElement(jsonData, ["data", "watchOverviewPage", "elements"]))
        totalItems = APIParser.getElement(jsonData, ["data", "watchOverviewPage", "pageInfo", "totalCount"])
        apiParams['offset'] += limit
        apiParams['limit'] = totalItems - limit
        res = self.getRequest(requestURL.format(**apiParams), self.token)
        jsonData = res.json()
        formats += APIParser.parseFormats(APIParser.getElement(jsonData, ["data", "watchOverviewPage", "elements"]))
        return formats

    def getLive(self):
        requestURL = f'{API.APIBASEURL}?operationName=LiveTvStations&variables={{"epgCount":4,"filter":{{"channelTypes":["BROADCAST"]}}}}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{API.ENDPOINTS["LiveTvStations"]}"}}}}'
        res = self.getRequest(requestURL, self.token)
        jsonData = res.json()
        format = APIParser.parseLiveChannels(jsonData)
        return format

    def getEvents(self):
        requestURL = f'{API.APIBASEURL}?operationName=LiveEventsOverviewPage&variables={{}}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{API.ENDPOINTS["LiveEventsOverviewPage"]}"}}}}'
        res = self.getRequest(requestURL, self.token)
        jsonData = res.json()
        format = APIParser.parseEvents(jsonData)
        return format

    def search(self, query):
        requestURL = f'{API.APIBASEURL}?operationName=Search&variables={{"input":{{"scopes":["MUSIC","PODCAST","WATCH","AUDIOBOOKS"],"term":"{query}"}}}}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{API.ENDPOINTS["Search"]}"}}}}'
        res = self.getRequest(requestURL, self.token)
        jsonData = res.json()

        for result in APIParser.getElement(jsonData, ["data", "search", "result"]):
            key = APIParser.getElement(result, ["titleKey"])
            if key == "WATCH":
                return APIParser.parseFormats(APIParser.getElement(result, ["items"]))

    def getRecommendationOverview(self, offset=0, take=15, cacheFile=None):
        requestURL = f'{API.APIBASEURL}?operationName=ExploreWidgetWatch&variables={{"area":"home","offset":{offset},"take":{take}}}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{API.ENDPOINTS["ExploreWidgetWatch"]}"}}}}'
        res = self.getRequest(requestURL, self.token)
        recommendations = []
        try:
            jsonData = res.json()
            for result in APIParser.getElement(jsonData, ["data", "editorialView", "widgets", "items"]):
                widgetName = APIParser.getElement(result, ["__typename"])
                if widgetName in ["TeaserWidget", "SavedFavoriteWidget", "WidgetContinue"]:
                    elements = APIParser.getElement(result, ["elements"])
                    if len(elements) > 0:
                        id = APIParser.getElement(result, ["id"])
                        title = APIParser.getElement(result, ["title"])
                        recommendations.append(APIParser.RecommendationOverview(id, title))
            if cacheFile != None:
                try:
                    os.remove(cacheFile)
                except OSError:
                    pass
                with open(cacheFile, "w", encoding='utf-8') as f:
                    f.write(res.content.decode())
        except:
            pass
        return recommendations

    def getRecommendation(self, recId, cacheFile=None):
        jsonData = {}
        formats = []
        if cacheFile == None:
            requestURL = f'{API.APIBASEURL}?operationName=ExploreWidgetWatch&variables={{"area":"home","offset":0,"take":15}}&extensions={{"persistedQuery":{{"version":1,"sha256Hash":"{API.ENDPOINTS["ExploreWidgetWatch"]}"}}}}'
            res = self.getRequest(requestURL, self.token)
            jsonData = res.json()
        else:
            with open(cacheFile, encoding='utf-8') as json_file:
                jsonData = json.load(json_file)
        for result in APIParser.getElement(jsonData, ["data", "editorialView", "widgets", "items"]):
            id = APIParser.getElement(result, ["id"], True)
            if id == recId:
                elements = APIParser.getElement(result, ["elements"])
                for element in elements:
                    if element == None:
                        continue
                    formats.append(APIParser.parseFormatOverview(element))
        return formats